import { expect } from 'chai';
import { describe } from 'mocha';
import { deployContractsForTests, getSigners } from './helpers/deploy-helper';
import { Signers } from './types';
import { OfferingToken } from '../../types';
import { upgradeTransparentProxyContract } from '../../scripts/upgradeTransparentProxyContract';

describe('20 - Offering Token Contract Test', function () {
  let contract: OfferingToken;
  let signers: Signers;
  beforeEach(async function () {
    contract = (await deployContractsForTests()).offeringTokenContract;
    signers = await getSigners();
  });
  describe('ERC 721 standard tests:', function () {
    it('contract can be deployed', async function () {
      await expect(contract).to.exist;
    });

    it('token should have name OfferingToken', async function () {
      await expect(await contract.name()).is.equal('OfferingToken');
    });

    it('token should have OFRNG symbol', async function () {
      await expect(await contract.symbol()).is.equal('OFRNG');
    });
  });
  describe('contract extends OpenZeppelin Ownable ', function () {
    it('should check contract owner', async function () {
      expect(await contract.owner()).to.equal(signers.admin.address);
    });
    it('owner can change the owner', async function () {
      await contract.transferOwnership(signers.signer1.address);
      expect(await contract.owner()).to.equal(signers.signer1.address);
    });
  });
  describe('token transfer tests when admin add asset', function () {
    beforeEach(async function () {
      await contract
        .connect(signers.admin)
        .addAsset(
          'unique-assetid-1',
          'unique-oid-1',
          'https://example.com/token-uri-1',
          signers.admin.address,
          100,
          1234567890,
          '10',
          '100MB',
          'target1',
          'sl1',
          '0x00',
        );
      await contract
        .connect(signers.admin)
        .addAsset(
          'unique-assetid-2',
          'unique-oid-2',
          'https://example.com/token-uri-2',
          signers.admin.address,
          100,
          1234567890,
          '10',
          '100MB',
          'target2',
          'sl2',
          '0x00',
        );
    });

    it('we can get asset id (hashed number based on oid) from offering id for two tokens minted', async function () {
      const offeringTokenIndx1 = await contract.getIDHash('unique-oid-1');
      const offeringTokenIndx2 = await contract.getIDHash('unique-oid-2');
      expect(offeringTokenIndx1).to.equal(
        '42719231677886629230429641141315469974786051051324644368487231673545418757415',
      );
      expect(offeringTokenIndx2).to.equal(
        '45286243120950074712315413008989232187669010308233156361261391424689958769314',
      );
    });
    it('getIDHash will generate token index even if token was not minted but revert with invalid token ID', async function () {
      const offeringTokenIndx3 = await contract.getIDHash('unique-oid-3');
      expect(offeringTokenIndx3.toString()).to.equal(
        '6194704066268248772908610227858564140428729439241732440098170692922599136581',
      );
    });
    it('owner of method wil revert if token id is not recognized with ERC721: invalid token ID', async function () {
      const offeringTokenIndx3 = await contract.getIDHash('unique-oid-3');
      await expect(contract.ownerOf(offeringTokenIndx3)).to.be.revertedWith(
        'ERC721: invalid token ID',
      );
    });

    it('offering token should be minted and be in account of the admin', async function () {
      const offeringTokenIndx1 = await contract.getIDHash('unique-oid-1');
      const offeringTokenIndx2 = await contract.getIDHash('unique-oid-2');
      expect(await contract.ownerOf(offeringTokenIndx1)).to.equal(
        signers.admin.address,
      );
      expect(await contract.ownerOf(offeringTokenIndx2)).to.equal(
        signers.admin.address,
      );
    });

    it('should transfer token from owner to another address', async function () {
      const offeringTokenIndx1 = contract.getIDHash('unique-oid-1');
      await contract
        .connect(signers.admin)
        .transferFrom(
          signers.admin.address,
          signers.signer1.address,
          offeringTokenIndx1,
        );
      expect(await contract.ownerOf(offeringTokenIndx1)).to.equal(
        signers.signer1.address,
      );
    });

    it('should not allow transfer from non-owner', async function () {
      const offeringTokenIndx1 = contract.getIDHash('unique-oid-1');
      await expect(
        contract
          .connect(signers.signer2)
          .transferFrom(
            signers.admin.address,
            signers.signer2.address,
            offeringTokenIndx1,
          ),
      ).to.be.revertedWith('ERC721: caller is not token owner or approved');
    });

    it('should approve and then transfer token by approved address', async function () {
      const offeringTokenIndx1 = contract.getIDHash('unique-oid-1');
      await contract
        .connect(signers.admin)
        .approve(signers.signer2.address, offeringTokenIndx1);
      await contract
        .connect(signers.signer2)
        .transferFrom(
          signers.admin.address,
          signers.signer2.address,
          offeringTokenIndx1,
        );
      expect(await contract.ownerOf(offeringTokenIndx1)).to.equal(
        signers.signer2.address,
      );
    });

    it('should emit Transfer event on token transfer', async function () {
      const offeringTokenIndx = await contract.getIDHash('unique-oid-1');
      await expect(
        contract
          .connect(signers.admin)
          .transferFrom(
            signers.admin.address,
            signers.signer1.address,
            offeringTokenIndx,
          ),
      )
        .to.emit(contract, 'Transfer')
        .withArgs(
          signers.admin.address,
          signers.signer1.address,
          offeringTokenIndx,
        );
    });

    it('should upgrade contract without losing token data', async function () {
      const offeringTokenIndx = await contract.getIDHash('unique-oid-1');
      // Check initial token
      expect(await contract.ownerOf(offeringTokenIndx)).to.equal(
        signers.admin.address,
      );
      expect(await contract.ver()).to.equal(1);

      // Upgrade contract
      await upgradeTransparentProxyContract({
        contractName: 'OfferingTokenV2',
        proxyContractAddress: contract.address,
      });
      expect(await contract.ownerOf(offeringTokenIndx)).to.equal(
        signers.admin.address,
      );
      expect(await contract.ver()).to.equal(2);
    });
  });
});
